Import-Module "$PSScriptRoot\UniversalDashboard.MaterialUI.psm1" -ErrorAction SilentlyContinue

if (-not $IsCoreCLR) {
    [PowerShellUniversal.Cmdlets.AssemblyResolver]::Bind()
}

$TAType = [psobject].Assembly.GetType('System.Management.Automation.TypeAccelerators')
$TAtype::Add('File', 'PowerShellUniversal.PSUFile')
$TAtype::Add('Documentation', 'PowerShellUniversal.DocumentationAttribute')
$TAtype::Add('Component', 'PowerShellUniversal.ComponentAttribute')

function Start-PSUServer {
    [CmdletBinding(DefaultParameterSetName = "Service")]
    param(
        [Parameter(ParameterSetName = 'Path')]
        [string]$ExecutablePath,
        [Parameter(ParameterSetName = 'Path')]
        [string]$ListenAddress,
        [Parameter(ParameterSetName = 'Path')]
        [int]$Port,
        [Parameter(ParameterSetName = 'Path')]
        [ScriptBlock]$Configuration
    )
    
    if ($MyInvocation.BoundParameters.Count -eq 0)
    {
        Start-Service 'PowerShellUniversal'
        return
    }

    if ([UniversalAutomation.RemoteCommand]::Configuration) {
        & $Configuration
        return
    }

    if (-not $PSBoundParameters.ContainsKey("ExecutablePath")) {
        $ExecutablePath = "Universal.Server"
        if ($PSVersionTable.PSEdition -eq 'Desktop' -or $IsWindows) {
            $ExecutablePath = "Universal.Server.exe"
        }
    }

    $Command = Get-Command $ExecutablePath -ErrorAction SilentlyContinue
    if ($null -eq $Command) {
        $ExecutablePath = Join-Path $PSScriptRoot $ExecutablePath
        $Command = Get-Command $ExecutablePath -ErrorAction SilentlyContinue
        if ($null -eq $Command) {
            throw 'Unable to locate the Universal Server executable. You can use Install-PSUServer the server for your platform. Use the -AddToPath parameter to add the installation directory the the PATH.'
        }
    }

    if ($PSVersionTable.PSEdition -ne 'Desktop' -and -not $IsWindows) {
        try {
            chmod +x $ExecutablePath
        }
        catch {
            Write-Warning "Failed to set executable flag. You may have to run 'chmod +x' yourself on $ExecutablePath. $_"
        }
    }

    if ($ListenAddress) {
        $Env:Kestrel__Endpoints__HTTP__Url = $ListenAddress
    }
    elseif ($PSBoundParameters.ContainsKey("Port")) {
        $Env:Kestrel__Endpoints__HTTP__Url = "http://*:$port"
    }

    if ($Configuration) {
        $scriptName = (Get-PSCallStack | Select-Object -Last 1).ScriptName
        if (-not $scriptName) {
            $scriptName = (Get-PSCallStack | Select-Object -Last 1 -Skip 1).ScriptName
        }
        $Env:Data__ConfigurationScript = $scriptName
    }

    $Process = Start-Process -FilePath $ExecutablePath -PassThru

    $Process
}

function Install-PSUServer {
    <#
    .SYNOPSIS
    Install the PowerShell Universal server.
    
    .DESCRIPTION
    Install the PowerShell Universal server. This is a convenience function that will install the server for your platform. On Windows, it will install the 
    server as a Windows service. On Linux, it will install the server as a systemd service. On Mac, it will install the server as a launchd service.
    
    .PARAMETER Path
    The path to store the PowerShell Universal binaries. If not specified, the default installation path will be used.
    
    .PARAMETER AddToPath
    Whether to add the path to the PATH environment variable.
    
    .PARAMETER Version
    The version of PowerShell Universal to install.
    
    .PARAMETER LatestVersion
    Install the most recent version. 
    
    .EXAMPLE
    Install-PSUServer
    #>
    [CmdletBinding(DefaultParameterSetName = "Version")]
    param(
        [Parameter()]
        [string]$Path,
        [Parameter(ParameterSetName = "Version")]
        [string]$Version = (Get-Module Universal).Version,
        [Parameter(ParameterSetName = "Latest")]
        [Switch]$LatestVersion,
        [Parameter()]
        [string]$IISWebsite,
        [Parameter()]
        [string]$IISAppPool = "PowerShellUniversal",
        [Parameter()]
        [int]$IISPort
    )

    if ($IISWebsite -and ($IsLinux -or $IsMacOS)) {
        throw "IISWebsite is only supported on Windows."
    }

    if ($IISWebsite) {
        Import-Module WebAdministration -ErrorAction Stop
    }
    
    if ($platform -eq 'win-x64' -and -not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
        throw 'You must be an administrator to install the Universal Server. Please run the command as an administrator.'
    }

    $platform = "win-x64";
    $folder = 'CommonApplicationData'
    if ($PSVersionTable.PSEdition -eq 'Core') {
        if ($IsLinux) {
            $platform = "linux-x64"
        }
        elseif ($IsMacOS) {
            $folder = 'ApplicationData'
            $platform = "osx-x64"
        }
    }

    if (-not $Path -and -not $IISWebsite) {
        $ProgramData = [System.Environment]::GetFolderPath($folder)
        $Path = [System.IO.Path]::Combine($ProgramData, "PowerShellUniversal", "Server")
    }

    if (-not $Path -and $IISWebsite) {
        $Path = "C:\inetpub\wwwroot\PowerShellUniversal"
        New-Item $Path -ItemType Directory | Out-Null
    }
    
    if ($LatestVersion) {
        $Version = (Invoke-WebRequest https://imsreleases.blob.core.windows.net/universal/production/v5-version.txt).Content
    }

    Write-Verbose "Downloading version $Version"

    if (($PSVersionTable.PSEdition -eq 'Desktop' -or $IsWindows) -and -not $IISWebsite) {
        $Temp = [System.IO.Path]::GetTempPath()
        $Msi = (Join-Path $Temp "Universal.$Version.$platform.msi")
        Remove-Item $Msi -Force -ErrorAction SilentlyContinue
        Invoke-WebRequest "https://imsreleases.blob.core.windows.net/universal/production/$version/PowerShellUniversal.$Version.msi" -OutFile $Msi  
        Write-Verbose "Download complete. Installing from MSI."
        Start-Process msiexec.exe -ArgumentList "/i $Msi /quiet /qn /norestart" -Wait
    } else {
        Write-Verbose "Installing server to $Path"
        $Temp = [System.IO.Path]::GetTempPath()
        $Zip = (Join-Path $Temp "Universal.$Version.$platform.zip")
        Remove-Item $Zip -Force -ErrorAction SilentlyContinue
        Invoke-WebRequest "https://imsreleases.blob.core.windows.net/universal/production/$version/Universal.$platform.$Version.zip" -OutFile $Zip
        Write-Verbose "Download complete. Unzipping to $Path"
        Expand-Archive -Path $Zip -DestinationPath $Path -Force
        Remove-Item $Zip -Force
    }

    if ($IISWebsite) {
        New-WebAppPool -Name $IISAppPool | Out-Null
        New-Website -Name $IISWebsite -Port $IISPort -PhysicalPath $Path -ApplicationPool $IISAppPool | Out-Null
        Start-Website -Name $IISWebsite
    }

    if ($IsMacOS -or $IsLinux) {
        $ServerPath = Join-Path $Path "Universal.Server"
        /bin/chmod +x $ServerPath
    }

    if ($IsLinux) {
        Write-Verbose "Creating and starting PowerShellUniversal service"
        touch /etc/systemd/system/PowerShellUniversal.service
        chmod 664 /etc/systemd/system/PowerShellUniversal.service
        "[Unit]
        Description=PowerShell Universal
        
        [Service]
        ExecStart=$Path/Universal.Server
        
        [Install]
        WantedBy=multi-user.target" | Out-File /etc/systemd/system/PowerShellUniversal.service

        systemctl daemon-reload
        systemctl start PowerShellUniversal
        systemctl enable PowerShellUniversal
    }
    
    if ($PSVersionTable.PSEdition -eq 'Desktop' -or $IsWindows) {
    }
    else {
        Write-Verbose "Adding $Path to `$PATH variable"
        $PathSeparator = ":"
        $envPath = [Environment]::GetEnvironmentVariable('PATH')
        $newpath = $envPath + $PathSeparator + $Path
        [Environment]::SetEnvironmentVariable("PATH", $newpath)
    }
    
    $Env:PATH += $PathSeparator + $Path
    
}

function Update-PSUServer {
    <#
    .SYNOPSIS
    Update the PowerShell Universal server.
    
    .DESCRIPTION
    Update the PowerShell Universal server. This is a convenience function that will update the server for your platform. 
    
    .PARAMETER Path
    The path for the PowerShell Universal binaries. If not specified, the path will attempt to be resolved.
    
    .PARAMETER Version
    The version to upgrade to. 
    
    .PARAMETER LatestVersion
    Upgrade to the latest version. 
    
    .EXAMPLE
    Update-PSUServer
    #>
    [CmdletBinding(DefaultParameterSetName = "Version")]
    param(
        [Parameter()]
        [string]$Path,
        [Parameter(ParameterSetName = "Version")]
        [string]$Version = (Get-Module Universal).Version,
        [Parameter(ParameterSetName = "Latest")]
        [Switch]$LatestVersion,
        [Parameter()]
        [string]$IISWebsite
    )
    
    $platform = "win-x64";
    if ($PSVersionTable.PSEdition -eq 'Core') {
        if ($IsLinux) {
            $platform = "linux-x64"
        }
        elseif ($IsMacOS) {
            $platform = "osx-x64"
        }
    }

    if ($platform -eq 'win-x64' -and -not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
        throw 'You must be an administrator to update the Universal Server. Please run the command as an administrator.'
    }

    if ($IISWebsite -and ($IsLinux -or $IsMacOS)) {
        throw "IISWebsite is only supported on Windows."
    }

    if ($IISWebsite) {
        Import-Module WebAdministration -ErrorAction Stop
    }
    
    if (-not $Path -and -not $IISWebsite) {
        if ($PSVersionTable.PSEdition -eq 'Desktop' -or $IsWindows) {
            $ServerPath = Get-Command "Universal.Server.exe" -ErrorAction SilentlyContinue
        }
        else {
            $ServerPath = Get-Command "Universal.Server" -ErrorAction SilentlyContinue
        }

        if (-not $ServerPath) {
            throw "Unable to locate existing PowerShell Universal installation. Use the -Path parameter to specify the folder of the previous installation."
        }

        $Path = [System.IO.Path]::GetDirectoryName($ServerPath.Source)
    }

    if (-not $Path -and $IISWebsite) {
        $Path = (Get-Website -Name $IISWebsite).PhysicalPath
    }
    
    if ($LatestVersion) {
        $Version = (Invoke-WebRequest https://imsreleases.blob.core.windows.net/universal/production/v5-version.txt).Content
    }

    if (($PSVersionTable.PSEdition -eq 'Desktop' -or $IsWindows) -and -not $IISWebsite) {
        $Temp = [System.IO.Path]::GetTempPath()
        $Msi = (Join-Path $Temp "Universal.$Version.$platform.msi")
        Remove-Item $Msi -Force -ErrorAction SilentlyContinue
        Invoke-WebRequest "https://imsreleases.blob.core.windows.net/universal/production/$version/PowerShellUniversal.$Version.msi" -OutFile $Msi
        Write-Verbose "Download complete. Installing from MSI."
        Start-Process msiexec.exe -ArgumentList "/i $Msi /quiet /qn /norestart" -Wait
    } else {
        Write-Verbose "Upgrading server installed at $Path"

        Write-Verbose "Downloading version $Version"

        $Temp = [System.IO.Path]::GetTempPath()
        $Zip = (Join-Path $Temp "Universal.$Version.$platform.zip")
        Remove-Item $Zip -Force -ErrorAction SilentlyContinue

        if ($IISWebsite) {
            $AppPool = (Get-Website -Name $IISWebsite).ApplicationPool
            Stop-Website -Name $IISWebsite
            Stop-WebAppPool -Name $AppPool
        }

        if ($IsLinux) {
            Write-Verbose "Stopped PowerShellUniversal service"
            systemctl stop PowerShellUniversal
            systemctl disable PowerShellUniversal
        }

        Remove-Item $Path -Force -Recurse
        Invoke-WebRequest "https://imsreleases.blob.core.windows.net/universal/production/$version/Universal.$platform.$Version.zip" -OutFile $Zip

        Write-Verbose "Download complete. Extracting to $Path"

        Expand-Archive -Path $Zip -DestinationPath $Path -Force
        Remove-Item $Zip -Force
        
        if ($IISWebsite) {
            Get-ChildItem $Path -Recurse | Unblock-File
            Start-Website -Name $IISWebsite
        }

        if ($IsMacOS -or $IsLinux) {
            $ServerPath = Join-Path $Path "Universal.Server"
            /bin/chmod +x $ServerPath
        }

        if ($IsLinux) {
            Write-Verbose "Started PowerShellUniversal service"
            systemctl start PowerShellUniversal
            systemctl enable PowerShellUniversal
        }
    }
}

function Uninstall-PSUServer {
    <#
    .SYNOPSIS
    Removes the PowerShell Universal server.
    
    .DESCRIPTION
    Removes the PowerShell Universal server. This is a convenience function that will remove the server for your platform.
    
    .PARAMETER Path
    The path to the PowerShell Universal binaries. If not specified, the path will attempt to be resolved.
    
    .EXAMPLE
    Remove-PSUServer
    #>
    [Alias("Remove-PSUServer")]
    [CmdletBinding(DefaultParameterSetName = "Version")]
    param(
        [Parameter()]
        [string]$Path,
        [Parameter()]
        [string]$IISWebsite
    )

    if (($PSVersionTable.PSEdition -eq 'Desktop' -or $IsWindows) -and -not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
        throw 'You must be an administrator to remove the Universal Server. Please run the command as an administrator.'
    }

    if ($IISWebsite -and ($IsLinux -or $IsMacOS)) {
        throw "IISWebsite is only supported on Windows."
    }

    if ($IISWebsite) {
        Import-Module WebAdministration -ErrorAction Stop
    }

    if (($PSVersionTable.PSEdition -eq 'Desktop' -or $IsWindows) -and -not $IISWebsite)
    {
        Write-Verbose "Locating local package with Win32_Product"
        $Package = Get-CimInstance win32_product  -Filter "Name like 'PowerShell Universal%'" | Select-Object -First 1
        if ($Package)
        {
            Write-Verbose "Removing via msiexec."
            Start-Process msiexec.exe -ArgumentList "/x $( $Package.LocalPackage ) /quiet /qn /norestart" -Wait
        } else {
            throw "Unable to locate PowerShell Universal installation."
        }
    } 
    else 
    {
        if (-not $Path -and -not $IISWebsite) {
            $ServerPath = Get-Command "Universal.Server" -ErrorAction SilentlyContinue

            if (-not $ServerPath) {
                throw "Unable to locate existing PowerShell Universal installation. Use the -Path parameter to specify the folder of the previous installation."
            }

            $Path = [System.IO.Path]::GetDirectoryName($ServerPath.Source)
        }

        if ($IISWebsite) {
            $Path = (Get-Website -Name $IISWebsite).PhysicalPath
            $AppPool = (Get-Website -Name $IISWebsite).ApplicationPool
            Stop-Website -Name $IISWebsite
            Remove-Website -Name $IISWebsite
            Remove-WebAppPool $AppPool

            Write-Verbose "Removing application files"
            Remove-Item $Path -Force -Recurse
        }

        if ($IsLinux) {
            Write-Verbose "Stopped PowerShellUniversal service"
            systemctl stop PowerShellUniversal
            systemctl disable PowerShellUniversal
            Remove-Item /etc/systemd/system/PowerShellUniversal.service -Force
            systemctl daemon-reload

            Write-Verbose "Removing application files"
            Remove-Item $Path -Force -Recurse
        }
    }
}
# SIG # Begin signature block
# MIIubQYJKoZIhvcNAQcCoIIuXjCCLloCAQMxDTALBglghkgBZQMEAgEwewYKKwYB
# BAGCNwIBBKBtBGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCC9Wn0Y1luS2p4e
# tQlYAILkz2xE77k0wOy6xkc4sb95NqCCE+wwggWiMIIEiqADAgECAhB4AxhCRXCK
# Qc9vAbjutKlUMA0GCSqGSIb3DQEBDAUAMEwxIDAeBgNVBAsTF0dsb2JhbFNpZ24g
# Um9vdCBDQSAtIFIzMRMwEQYDVQQKEwpHbG9iYWxTaWduMRMwEQYDVQQDEwpHbG9i
# YWxTaWduMB4XDTIwMDcyODAwMDAwMFoXDTI5MDMxODAwMDAwMFowUzELMAkGA1UE
# BhMCQkUxGTAXBgNVBAoTEEdsb2JhbFNpZ24gbnYtc2ExKTAnBgNVBAMTIEdsb2Jh
# bFNpZ24gQ29kZSBTaWduaW5nIFJvb3QgUjQ1MIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEAti3FMN166KuQPQNysDpLmRZhsuX/pWcdNxzlfuyTg6qE9aND
# m5hFirhjV12bAIgEJen4aJJLgthLyUoD86h/ao+KYSe9oUTQ/fU/IsKjT5GNswWy
# KIKRXftZiAULlwbCmPgspzMk7lA6QczwoLB7HU3SqFg4lunf+RuRu4sQLNLHQx2i
# CXShgK975jMKDFlrjrz0q1qXe3+uVfuE8ID+hEzX4rq9xHWhb71hEHREspgH4nSr
# /2jcbCY+6R/l4ASHrTDTDI0DfFW4FnBcJHggJetnZ4iruk40mGtwEd44ytS+ocCc
# 4d8eAgHYO+FnQ4S2z/x0ty+Eo7+6CTc9Z2yxRVwZYatBg/WsHet3DUZHc86/vZWV
# 7Z0riBD++ljop1fhs8+oWukHJZsSxJ6Acj2T3IyU3ztE5iaA/NLDA/CMDNJF1i7n
# j5ie5gTuQm5nfkIWcWLnBPlgxmShtpyBIU4rxm1olIbGmXRzZzF6kfLUjHlufKa7
# fkZvTcWFEivPmiJECKiFN84HYVcGFxIkwMQxc6GYNVdHfhA6RdktpFGQmKmgBzfE
# ZRqqHGsWd/enl+w/GTCZbzH76kCy59LE+snQ8FB2dFn6jW0XMr746X4D9OeHdZrU
# SpEshQMTAitCgPKJajbPyEygzp74y42tFqfT3tWbGKfGkjrxgmPxLg4kZN8CAwEA
# AaOCAXcwggFzMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDAzAP
# BgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQfAL9GgAr8eDm3pbRD2VZQu86WOzAf
# BgNVHSMEGDAWgBSP8Et/qC5FJK5NUPpjmove4t0bvDB6BggrBgEFBQcBAQRuMGww
# LQYIKwYBBQUHMAGGIWh0dHA6Ly9vY3NwLmdsb2JhbHNpZ24uY29tL3Jvb3RyMzA7
# BggrBgEFBQcwAoYvaHR0cDovL3NlY3VyZS5nbG9iYWxzaWduLmNvbS9jYWNlcnQv
# cm9vdC1yMy5jcnQwNgYDVR0fBC8wLTAroCmgJ4YlaHR0cDovL2NybC5nbG9iYWxz
# aWduLmNvbS9yb290LXIzLmNybDBHBgNVHSAEQDA+MDwGBFUdIAAwNDAyBggrBgEF
# BQcCARYmaHR0cHM6Ly93d3cuZ2xvYmFsc2lnbi5jb20vcmVwb3NpdG9yeS8wDQYJ
# KoZIhvcNAQEMBQADggEBAKz3zBWLMHmoHQsoiBkJ1xx//oa9e1ozbg1nDnti2eEY
# XLC9E10dI645UHY3qkT9XwEjWYZWTMytvGQTFDCkIKjgP+icctx+89gMI7qoLao8
# 9uyfhzEHZfU5p1GCdeHyL5f20eFlloNk/qEdUfu1JJv10ndpvIUsXPpYd9Gup7EL
# 4tZ3u6m0NEqpbz308w2VXeb5ekWwJRcxLtv3D2jmgx+p9+XUnZiM02FLL8Mofnre
# kw60faAKbZLEtGY/fadY7qz37MMIAas4/AocqcWXsojICQIZ9lyaGvFNbDDUswar
# AGBIDXirzxetkpNiIHd1bL3IMrTcTevZ38GQlim9wX8wggbmMIIEzqADAgECAhB3
# vQ4DobcI+FSrBnIQ2QRHMA0GCSqGSIb3DQEBCwUAMFMxCzAJBgNVBAYTAkJFMRkw
# FwYDVQQKExBHbG9iYWxTaWduIG52LXNhMSkwJwYDVQQDEyBHbG9iYWxTaWduIENv
# ZGUgU2lnbmluZyBSb290IFI0NTAeFw0yMDA3MjgwMDAwMDBaFw0zMDA3MjgwMDAw
# MDBaMFkxCzAJBgNVBAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMS8w
# LQYDVQQDEyZHbG9iYWxTaWduIEdDQyBSNDUgQ29kZVNpZ25pbmcgQ0EgMjAyMDCC
# AiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBANZCTfnjT8Yj9GwdgaYw90g9
# z9DljeUgIpYHRDVdBs8PHXBg5iZU+lMjYAKoXwIC947Jbj2peAW9jvVPGSSZfM8R
# Fpsfe2vSo3toZXer2LEsP9NyBjJcW6xQZywlTVYGNvzBYkx9fYYWlZpdVLpQ0LB/
# okQZ6dZubD4Twp8R1F80W1FoMWMK+FvQ3rpZXzGviWg4QD4I6FNnTmO2IY7v3Y2F
# QVWeHLw33JWgxHGnHxulSW4KIFl+iaNYFZcAJWnf3sJqUGVOU/troZ8YHooOX1Re
# veBbz/IMBNLeCKEQJvey83ouwo6WwT/Opdr0WSiMN2WhMZYLjqR2dxVJhGaCJedD
# CndSsZlRQv+hst2c0twY2cGGqUAdQZdihryo/6LHYxcG/WZ6NpQBIIl4H5D0e6lS
# TmpPVAYqgK+ex1BC+mUK4wH0sW6sDqjjgRmoOMieAyiGpHSnR5V+cloqexVqHMRp
# 5rC+QBmZy9J9VU4inBDgoVvDsy56i8Te8UsfjCh5MEV/bBO2PSz/LUqKKuwoDy3K
# 1JyYikptWjYsL9+6y+JBSgh3GIitNWGUEvOkcuvuNp6nUSeRPPeiGsz8h+WX4VGH
# aekizIPAtw9FbAfhQ0/UjErOz2OxtaQQevkNDCiwazT+IWgnb+z4+iaEW3VCzYkm
# eVmda6tjcWKQJQ0IIPH/AgMBAAGjggGuMIIBqjAOBgNVHQ8BAf8EBAMCAYYwEwYD
# VR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQU
# 2rONwCSQo2t30wygWd0hZ2R2C3gwHwYDVR0jBBgwFoAUHwC/RoAK/Hg5t6W0Q9lW
# ULvOljswgZMGCCsGAQUFBwEBBIGGMIGDMDkGCCsGAQUFBzABhi1odHRwOi8vb2Nz
# cC5nbG9iYWxzaWduLmNvbS9jb2Rlc2lnbmluZ3Jvb3RyNDUwRgYIKwYBBQUHMAKG
# Omh0dHA6Ly9zZWN1cmUuZ2xvYmFsc2lnbi5jb20vY2FjZXJ0L2NvZGVzaWduaW5n
# cm9vdHI0NS5jcnQwQQYDVR0fBDowODA2oDSgMoYwaHR0cDovL2NybC5nbG9iYWxz
# aWduLmNvbS9jb2Rlc2lnbmluZ3Jvb3RyNDUuY3JsMFYGA1UdIARPME0wQQYJKwYB
# BAGgMgEyMDQwMgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29t
# L3JlcG9zaXRvcnkvMAgGBmeBDAEEATANBgkqhkiG9w0BAQsFAAOCAgEACIhyJsav
# +qxfBsCqjJDa0LLAopf/bhMyFlT9PvQwEZ+PmPmbUt3yohbu2XiVppp8YbgEtfjr
# y/RhETP2ZSW3EUKL2Glux/+VtIFDqX6uv4LWTcwRo4NxahBeGQWn52x/VvSoXMNO
# Ca1Za7j5fqUuuPzeDsKg+7AE1BMbxyepuaotMTvPRkyd60zsvC6c8YejfzhpX0FA
# Z/ZTfepB7449+6nUEThG3zzr9s0ivRPN8OHm5TOgvjzkeNUbzCDyMHOwIhz2hNab
# XAAC4ShSS/8SS0Dq7rAaBgaehObn8NuERvtz2StCtslXNMcWwKbrIbmqDvf+28rr
# vBfLuGfr4z5P26mUhmRVyQkKwNkEcUoRS1pkw7x4eK1MRyZlB5nVzTZgoTNTs/Z7
# KtWJQDxxpav4mVn945uSS90FvQsMeAYrz1PYvRKaWyeGhT+RvuB4gHNU36cdZytq
# tq5NiYAkCFJwUPMB/0SuL5rg4UkI4eFb1zjRngqKnZQnm8qjudviNmrjb7lYYuA2
# eDYB+sGniXomU6Ncu9Ky64rLYwgv/h7zViniNZvY/+mlvW1LWSyJLC9Su7UpkNpD
# R7xy3bzZv4DB3LCrtEsdWDY3ZOub4YUXmimi/eYI0pL/oPh84emn0TCOXyZQK8ei
# 4pd3iu/YTT4m65lAYPM8Zwy2CHIpNVOBNNwwggdYMIIFQKADAgECAgwXNbgc0YbW
# 5XJW2qIwDQYJKoZIhvcNAQELBQAwWTELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEds
# b2JhbFNpZ24gbnYtc2ExLzAtBgNVBAMTJkdsb2JhbFNpZ24gR0NDIFI0NSBDb2Rl
# U2lnbmluZyBDQSAyMDIwMB4XDTI1MDcwMjE0MTk1MloXDTI4MDgwMTE0MTk1Mlow
# gZ0xCzAJBgNVBAYTAlVTMRIwEAYDVQQIEwlXaXNjb25zaW4xEDAOBgNVBAcTB01h
# ZGlzb24xHTAbBgNVBAoTFElyb25tYW4gU29mdHdhcmUgTExDMR0wGwYDVQQDExRJ
# cm9ubWFuIFNvZnR3YXJlIExMQzEqMCgGCSqGSIb3DQEJARYbc3VwcG9ydEBpcm9u
# bWFuc29mdHdhcmUuY29tMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA
# ttKDkouC6+LoaUb9aX/gz8ovLCfyAvc+9o543qnoVwkMyChpZa9QtbDpo06YEHAv
# A6gjBK2gjJayD8sJGi+eH1yEpyDdXiPd7C+uodSiS0ir6ObjR9lImRrMyVjNSvaH
# OkidzrshrCiifoLs9RN4uwO7P85pc5DWspAfJutY4lbUqx1DDWuuQ6Dd4oBkBr8Y
# lVAQvNBJWMtUWBFCJFVDhQgZh2igbT2UiYjwHWm4qOXmZm7JAPb2BxU2ho92Ph6T
# M54ab/gguQj+4zCyql/Iy133BpbBizlv9tY/O2PEZC3v8Sl4+Vv+SijsUd6fgDoc
# u2wlfIZOIK4U5EdMZC6MHYC6+jQ5veC/ve4XbuBnPsDEohsLwflR5Bnj7GrInW4h
# iSbNRyecfMdWbfvnGzP7+CwVW7kAwnAep+e9MwVJMCFLrUe7M2RcIRx310g/aNxY
# VJpCwn/uiohmXhgeQFLLTE057C0p8WkyabAgkMNVuXPjOFaNNwQuLgE4jm3E1lAG
# eEpRedXLBoyH0LIKdVkH508woGRGDo7gJX2LGr3jlDS+U8fW2eD3ZPElmqJtl4Ep
# YTmsKjA8oT85yHN3wkfGLDZK10WM2syAGC5voxfKFtuT5hOpZCWUyOhIy2uz7veb
# K9A2UTj8sqB6e8uKAmOoRxS/ndaqBfHgZ9ipthsPo5UCAwEAAaOCAdkwggHVMA4G
# A1UdDwEB/wQEAwIHgDCBmwYIKwYBBQUHAQEEgY4wgYswSgYIKwYBBQUHMAKGPmh0
# dHA6Ly9zZWN1cmUuZ2xvYmFsc2lnbi5jb20vY2FjZXJ0L2dzZ2NjcjQ1Y29kZXNp
# Z25jYTIwMjAuY3J0MD0GCCsGAQUFBzABhjFodHRwOi8vb2NzcC5nbG9iYWxzaWdu
# LmNvbS9nc2djY3I0NWNvZGVzaWduY2EyMDIwMFYGA1UdIARPME0wQQYJKwYBBAGg
# MgEyMDQwMgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3Jl
# cG9zaXRvcnkvMAgGBmeBDAEEATAJBgNVHRMEAjAAMEUGA1UdHwQ+MDwwOqA4oDaG
# NGh0dHA6Ly9jcmwuZ2xvYmFsc2lnbi5jb20vZ3NnY2NyNDVjb2Rlc2lnbmNhMjAy
# MC5jcmwwJgYDVR0RBB8wHYEbc3VwcG9ydEBpcm9ubWFuc29mdHdhcmUuY29tMBMG
# A1UdJQQMMAoGCCsGAQUFBwMDMB8GA1UdIwQYMBaAFNqzjcAkkKNrd9MMoFndIWdk
# dgt4MB0GA1UdDgQWBBTGcdd/VtnAqUmZQe3l0LERPIHjfTANBgkqhkiG9w0BAQsF
# AAOCAgEAuYLX2yaZNnZmRPqoVR/UB0z1GqAIwzq7diNMAhPgAQP4JaMXepUZBml/
# qJ0Klp9J/gSfPMkK6VRwHUOMiM+W7Q8oJdXy/beEvZ+2KYaFDbTWjQKl0FEuJY2Q
# fsw8tBR4qJlLqonHyqpeOINwU8UB6drhpjS7cRbQ76GVUxBFwiSQkh6XqwVFDljB
# bI8MSfBTGP493WrmgO+Jm+xwO1zhl8EO75Ov3yvU6XKJ/BFJnealGqqc2T9LcX9L
# jb58XGSFNVq/HvlnBceqUJFgerxNJmj6o/qVq4VA+D6QXzTBRBA+htO+d4zR6XJW
# RGVC5DWBEap769kNKtTNo9v9kO9DBZITgEmiLhMbVq6fyRc/sXzwV529/Nb/Jg+E
# aDoH3qGznjkia/iK2U+wV1OdKlggqe/8hTLDNmC0dhQW1P5XEFbPfWrAT/7rH4/Z
# zB0/RpM5bNRRu3STCpsyc2kUbMnq6rt9WbkweoA1EXIYGyDAiEgGwRpA4ACFl3CW
# PD0h/5SoC4ZhKKPzgOKdzBTv5/TZcpPsid3I934tLGk1n3MmGjpOR2y9CVU3Jzu4
# 78HJOjEe2pbaYxNDQ5x4fRzraJFB0bI2jDjYFgZBTBYPjJDs/DLxHmaAXIphbwNK
# mfdiJbEGuzEVJ5F8ASvA1yVigTZf7oOBEnME1oseU8yaKABmcsQxghnXMIIZ0wIB
# ATBpMFkxCzAJBgNVBAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMS8w
# LQYDVQQDEyZHbG9iYWxTaWduIEdDQyBSNDUgQ29kZVNpZ25pbmcgQ0EgMjAyMAIM
# FzW4HNGG1uVyVtqiMAsGCWCGSAFlAwQCAaB8MBAGCisGAQQBgjcCAQwxAjAAMBkG
# CSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEE
# AYI3AgEVMC8GCSqGSIb3DQEJBDEiBCAdOD+v8r4MK8PpN9qif3eXF+fGNiVUAh2e
# JqCT7QKJ5jALBgkqhkiG9w0BAQEEggIASkyGHERdFJ1MhS8Sj2efc2UbD9glyW7k
# 2Bw5DTU+VO7UomaDHddGXgNK2wDTOS5WuHm0c5jN7++7OC7Xuu6Dfc0J2L6UBjBJ
# kjeNZjEDxiymcYucRAf0IJxvovB3G3fj+s1pada1Jku5Ct7NzcfYmxZrC1B7eu6u
# T3pGrq+DjybWW+Q7UqfEcuMjhxEoy1zP3a5w0tYLiWzTQ0PaQ+Ieaprr8QTA3AjP
# tER/6K5/EluAPX15CmbIL6XRg0jswzz2aFx5bAPED9N6lylFa6mHwWEG18LamZPa
# 8fxq1BQ5XKLnXhzkqWJeg+6tlXEgWM6udfny34ofFlFEf9yyTg6yzCbmJP9Svzxw
# 5SqE1sewxC0/Irpm39pW62U6niLwf/5zZKZLyblR0nOzG57gv6wvmDic0fUDLtkm
# tQ6a4OC7ZW1SVtQbfd6vJKMN0doQTHjww7hNGEGq+EZEwV8AfB7kw62O3Wlwnsmm
# EKeLummAqW7TFnhWcNr5JOCpgVDCiAW15clilCAOHP9cQkY0uHw3ag4fz+jskhEA
# d7ikn951wxf8RwFk6Wl2xolFT60gnEvpLS6cPaMw4nmlB6oTPbFr+E0lqSI40FSN
# +k/iC0qnBpWIR6TXiSCpTOO/z3O9XHZC5awxpYlfXJBoYxOhr3BIpQY5kkJINfNC
# 0tZNzdJi3h+hghbFMIIWwQYKKwYBBAGCNwMDATGCFrEwghatBgkqhkiG9w0BBwKg
# ghaeMIIWmgIBAzENMAsGCWCGSAFlAwQCATCB6QYLKoZIhvcNAQkQAQSggdkEgdYw
# gdMCAQEGCysGAQQBoDICAwECMDEwDQYJYIZIAWUDBAIBBQAEIMQveUvwjZ1FFvXN
# I02IeiT3yVldoA8zz4WWD4QnSvF1AhRp+kXTyoEppcMl3mCnw8Tp53aIxRgPMjAy
# NTEwMjIxMjQ3NTZaMAMCAQECCG/S1L137BfXoFikVjBUMQswCQYDVQQGEwJCRTEZ
# MBcGA1UECgwQR2xvYmFsU2lnbiBudi1zYTEqMCgGA1UEAwwhR2xvYmFsc2lnbiBU
# U0EgZm9yIENvZGVTaWduMSAtIFI2oIISSzCCBYMwggNroAMCAQICDkXmuwODM8OF
# ZUjm/0VRMA0GCSqGSIb3DQEBDAUAMEwxIDAeBgNVBAsTF0dsb2JhbFNpZ24gUm9v
# dCBDQSAtIFI2MRMwEQYDVQQKEwpHbG9iYWxTaWduMRMwEQYDVQQDEwpHbG9iYWxT
# aWduMB4XDTE0MTIxMDAwMDAwMFoXDTM0MTIxMDAwMDAwMFowTDEgMB4GA1UECxMX
# R2xvYmFsU2lnbiBSb290IENBIC0gUjYxEzARBgNVBAoTCkdsb2JhbFNpZ24xEzAR
# BgNVBAMTCkdsb2JhbFNpZ24wggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoIC
# AQCVB+hzymb57BTKezz3DQjxtEULLIK0SMbrWzyug7hBkjMUpG9/6SrMxrCIa8W2
# idHGsv8UzlEUIexK3RtaxtaH7k06FQbtZGYLkoDKRN5zlE7zp4l/T3hjCMgSUG1C
# Zi9NuXkoTVIaihqAtxmBDn7EirxkTCEcQ2jXPTyKxbJm1ZCatzEGxb7ibTIGph75
# ueuqo7i/voJjUNDwGInf5A959eqiHyrScC5757yTu21T4kh8jBAHOP9msndhfuDq
# jDyqtKT285VKEgdt/Yyyic/QoGF3yFh0sNQjOvddOsqi250J3l1ELZDxgc1Xkvp+
# vFAEYzTfa5MYvms2sjnkrCQ2t/DvthwTV5O23rL44oW3c6K4NapF8uCdNqFvVIrx
# clZuLojFUUJEFZTuo8U4lptOTloLR/MGNkl3MLxxN+Wm7CEIdfzmYRY/d9XZkZeE
# CmzUAk10wBTt/Tn7g/JeFKEEsAvp/u6P4W4LsgizYWYJarEGOmWWWcDwNf3J2iiN
# GhGHcIEKqJp1HZ46hgUAntuA1iX53AWeJ1lMdjlb6vmlodiDD9H/3zAR+YXPM0j1
# ym1kFCx6WE/TSwhJxZVkGmMOeT31s4zKWK2cQkV5bg6HGVxUsWW2v4yb3BPpDW+4
# LtxnbsmLEbWEFIoAGXCDeZGXkdQaJ783HjIH2BRjPChMrwIDAQABo2MwYTAOBgNV
# HQ8BAf8EBAMCAQYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQUrmwFo5MT4qLn
# 4tcc1sfwf8hnU6AwHwYDVR0jBBgwFoAUrmwFo5MT4qLn4tcc1sfwf8hnU6AwDQYJ
# KoZIhvcNAQEMBQADggIBAIMl7ejR/ZVSzZ7ABKCRaeZc0ITe3K2iT+hHeNZlmKlb
# qDyHfAKK0W63FnPmX8BUmNV0vsHN4hGRrSMYPd3hckSWtJVewHuOmXgWQxNWV7Oi
# szu1d9xAcqyj65s1PrEIIaHnxEM3eTK+teecLEy8QymZjjDTrCHg4x362AczdlQA
# Iiq5TSAucGja5VP8g1zTnfL/RAxEZvLS471GABptArolXY2hMVHdVEYcTduZlu8a
# HARcphXveOB5/l3bPqpMVf2aFalv4ab733Aw6cPuQkbtwpMFifp9Y3s/0HGBfADo
# mK4OeDTDJfuvCp8ga907E48SjOJBGkh6c6B3ace2XH+CyB7+WBsoK6hsrV5twAXS
# e7frgP4lN/4Cm2isQl3D7vXM3PBQddI2aZzmewTfbgZptt4KCUhZh+t7FGB6ZKpp
# Q++Rx0zsGN1s71MtjJnhXvJyPs9UyL1n7KQPTEX/07kwIwdMjxC/hpbZmVq0mVcc
# pMy7FYlTuiwFD+TEnhmxGDTVTJ267fcfrySVBHioA7vugeXaX3yLSqGQdCWnsz5L
# yCxWvcfI7zjiXJLwefechLp0LWEBIH5+0fJPB1lfiy1DUutGDJTh9WZHeXfVVFsf
# rSQ3y0VaTqBESMjYsJnFFYQJ9tZJScBluOYacW6gqPGC6EU+bNYC1wpngwVayaQQ
# MIIGWTCCBEGgAwIBAgINAewckkDe/S5AXXxHdDANBgkqhkiG9w0BAQwFADBMMSAw
# HgYDVQQLExdHbG9iYWxTaWduIFJvb3QgQ0EgLSBSNjETMBEGA1UEChMKR2xvYmFs
# U2lnbjETMBEGA1UEAxMKR2xvYmFsU2lnbjAeFw0xODA2MjAwMDAwMDBaFw0zNDEy
# MTAwMDAwMDBaMFsxCzAJBgNVBAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52
# LXNhMTEwLwYDVQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAtIFNIQTM4
# NCAtIEc0MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA8ALiMCP64Bvh
# mnSzr3WDX6lHUsdhOmN8OSN5bXT8MeR0EhmW+s4nYluuB4on7lejxDXtszTHrMMM
# 64BmbdEoSsEsu7lw8nKujPeZWl12rr9EqHxBJI6PusVP/zZBq6ct/XhOQ4j+kxkX
# 2e4xz7yKO25qxIjw7pf23PMYoEuZHA6HpybhiMmg5ZninvScTD9dW+y279Jlz0UL
# VD2xVFMHi5luuFSZiqgxkjvyen38DljfgWrhsGweZYIq1CHHlP5CljvxC7F/f0aY
# Doc9emXr0VapLr37WD21hfpTmU1bdO1yS6INgjcZDNCr6lrB7w/Vmbk/9E818ZwP
# 0zcTUtklNO2W7/hn6gi+j0l6/5Cx1PcpFdf5DV3Wh0MedMRwKLSAe70qm7uE4Q6s
# bw25tfZtVv6KHQk+JA5nJsf8sg2glLCylMx75mf+pliy1NhBEsFV/W6RxbuxTAhL
# ntRCBm8bGNU26mSuzv31BebiZtAOBSGssREGIxnk+wU0ROoIrp1JZxGLguWtWoan
# Zv0zAwHemSX5cW7pnF0CTGA8zwKPAf1y7pLxpxLeQhJN7Kkm5XcCrA5XDAnRYZ4m
# iPzIsk3bZPBFn7rBP1Sj2HYClWxqjcoiXPYMBOMp+kuwHNM3dITZHWarNHOPHn18
# XpbWPRmwl+qMUJFtr1eGfhA3HWsaFN8CAwEAAaOCASkwggElMA4GA1UdDwEB/wQE
# AwIBhjASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBTqFsZp5+PLV0U5M6Tw
# QL7Qw71lljAfBgNVHSMEGDAWgBSubAWjkxPioufi1xzWx/B/yGdToDA+BggrBgEF
# BQcBAQQyMDAwLgYIKwYBBQUHMAGGImh0dHA6Ly9vY3NwMi5nbG9iYWxzaWduLmNv
# bS9yb290cjYwNgYDVR0fBC8wLTAroCmgJ4YlaHR0cDovL2NybC5nbG9iYWxzaWdu
# LmNvbS9yb290LXI2LmNybDBHBgNVHSAEQDA+MDwGBFUdIAAwNDAyBggrBgEFBQcC
# ARYmaHR0cHM6Ly93d3cuZ2xvYmFsc2lnbi5jb20vcmVwb3NpdG9yeS8wDQYJKoZI
# hvcNAQEMBQADggIBAH/iiNlXZytCX4GnCQu6xLsoGFbWTL/bGwdwxvsLCa0AOmAz
# HznGFmsZQEklCB7km/fWpA2PHpbyhqIX3kG/T+G8q83uwCOMxoX+SxUk+RhE7B/C
# pKzQss/swlZlHb1/9t6CyLefYdO1RkiYlwJnehaVSttixtCzAsw0SEVV3ezpSp9e
# FO1yEHF2cNIPlvPqN1eUkRiv3I2ZOBlYwqmhfqJuFSbqtPl/KufnSGRpL9KaoXL2
# 9yRLdFp9coY1swJXH4uc/LusTN763lNMg/0SsbZJVU91naxvSsguarnKiMMSME6y
# CHOfXqHWmc7pfUuWLMwWaxjN5Fk3hgks4kXWss1ugnWl2o0et1sviC49ffHykTAF
# nM57fKDFrK9RBvARxx0wxVFWYOh8lT0i49UKJFMnl4D6SIknLHniPOWbHuOqhIKJ
# PsBK9SH+YhDtHTD89szqSCd8i3VCf2vL86VrlR8EWDQKie2CUOTRe6jJ5r5IqitV
# 2Y23JSAOG1Gg1GOqg+pscmFKyfpDxMZXxZ22PLCLsLkcMe+97xTYFEBsIB3CLegL
# xo1tjLZx7VIh/j72n585Gq6s0i96ILH0rKod4i0UnfqWah3GPMrz2Ry/U02kR1l8
# lcRDQfkl4iwQfoH5DZSnffK1CfXYYHJAUJUg1ENEvvqglecgWbZ4xqRqqiKbMIIG
# YzCCBEugAwIBAgIQAQALIAWzlAdi2z42eZSbqTANBgkqhkiG9w0BAQwFADBbMQsw
# CQYDVQQGEwJCRTEZMBcGA1UEChMQR2xvYmFsU2lnbiBudi1zYTExMC8GA1UEAxMo
# R2xvYmFsU2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEzODQgLSBHNDAeFw0yNTA0
# MTExNDQ3MzlaFw0zNDEyMTAwMDAwMDBaMFQxCzAJBgNVBAYTAkJFMRkwFwYDVQQK
# DBBHbG9iYWxTaWduIG52LXNhMSowKAYDVQQDDCFHbG9iYWxzaWduIFRTQSBmb3Ig
# Q29kZVNpZ24xIC0gUjYwggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQCi
# W8SqTUnlI3g+4IXr5qoKPtXyrKinRGH3isuZjfsKfx6vzRKR8rN0kcmTdOpr1Bjh
# TBXqF51d9JvmCTRfJ3Lcw0SPWoT5miY8RyMLu5G0Z18H8+2eJ5YnOO09VSWYQ+tQ
# yygVTvCglJxFXVbr/p5+LYHJIdeyFnfRXUQNnaUpDQMrerdsjiwkT/mf91DLyIvi
# GNNpL2/RhDLlElj6jU1aHE/8ew26brs6visUOwYgTzmGDmviDcJcW6Kpa+U2Y9sI
# FaB3Gn88KpjQuJ7TgO4psPBF3qeubkAfmghEnIVSMu8bgc7QvVRdgy4nonUnM7jf
# xEDSYlG9TZnBsjrKIUdpLTHdKAVqS+EQsufi2bCeBg+P4UNfctpYeeeL8wYldHGc
# 3SlB1gmoRiEpSWswf37YqWd1zP/3JdgYTLBO3zx3fL5NQW9mUADQEw5gwytgVASm
# 2VGDYNivHmxa1fWq8LOTqwQEtFU8RsTSL4nTUguUHNGlwm6xlRDIYrZFVI4V9c8C
# AwEAAaOCAagwggGkMA4GA1UdDwEB/wQEAwIHgDAWBgNVHSUBAf8EDDAKBggrBgEF
# BQcDCDAdBgNVHQ4EFgQUgENM/LptHRPSdRKICXdv28ZPlVQwVgYDVR0gBE8wTTAI
# BgZngQwBBAIwQQYJKwYBBAGgMgEeMDQwMgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3
# Lmdsb2JhbHNpZ24uY29tL3JlcG9zaXRvcnkvMAwGA1UdEwEB/wQCMAAwgZAGCCsG
# AQUFBwEBBIGDMIGAMDkGCCsGAQUFBzABhi1odHRwOi8vb2NzcC5nbG9iYWxzaWdu
# LmNvbS9jYS9nc3RzYWNhc2hhMzg0ZzQwQwYIKwYBBQUHMAKGN2h0dHA6Ly9zZWN1
# cmUuZ2xvYmFsc2lnbi5jb20vY2FjZXJ0L2dzdHNhY2FzaGEzODRnNC5jcnQwHwYD
# VR0jBBgwFoAU6hbGaefjy1dFOTOk8EC+0MO9ZZYwQQYDVR0fBDowODA2oDSgMoYw
# aHR0cDovL2NybC5nbG9iYWxzaWduLmNvbS9jYS9nc3RzYWNhc2hhMzg0ZzQuY3Js
# MA0GCSqGSIb3DQEBDAUAA4ICAQC3psdKmXZ0/SBiJ72JfDcHPlfPCy+aJiKwSNuj
# JFmKf6OofbogWrt9Awis3lV1fwoaXmkR7bVvyyQ99Ry2KuijpBLmhYRKXiTNmjDa
# In+omfvpOS8v6/Z/iYeiUYsHDIObvh9evrxbrfG6JhKmdt/s1g/IFocjvMn8StJP
# hsDTRML5+0+21L4b8yE1RoqDfjJonxnoAQ2c90IHxUiu3+yp3pHWTz06PHWMvfpA
# EQ9SvZ20giVLcqLvgJTXlhO8b5fG31zAT7QbH+pFC/xhsX5Ryx2BhTiM1FkHSRNX
# cGx3eXIhaa3wNPhFCP0tDUgs6mob8LPB32r0YkvqNIl5KrUALhmhxUy6pxh38P0m
# kgwv0FD/CTmWAmSBcRzWBGA+wOxT3PPfonKY3X5iG/HRWvruOvRB+qTdXTPr/FWY
# QAkpp9N4h8P2FByGOIaTLDcQo2LIfIzfut0r8GxNLHSsHcwd2oT390dBfk1Q5AJZ
# S+GRbLXS7vXLMYpadsBZDo026r1+1zBgVOZzKjlhsXDN+TJL5jvz47262qCLpWgT
# Y3NUlu5yoYtm0kUnMYkPvF7+upQvBgd5Dcc9Kc0JqBvqOoF4VKddPHQZu5i6TeMD
# nV1A22yiBUAZcKoeyHrZBUCX+Bp1N9xFoykhYBzcI+6nfHX10MCmymZHI5XNcFS2
# yIJawjGCA0kwggNFAgEBMG8wWzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEdsb2Jh
# bFNpZ24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNpZ24gVGltZXN0YW1waW5nIENB
# IC0gU0hBMzg0IC0gRzQCEAEACyAFs5QHYts+NnmUm6kwCwYJYIZIAWUDBAIBoIIB
# LTAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwKwYJKoZIhvcNAQk0MR4wHDAL
# BglghkgBZQMEAgGhDQYJKoZIhvcNAQELBQAwLwYJKoZIhvcNAQkEMSIEIJ7MCGAn
# foFK0O4drUEJpOtDm9E0C8MgD9Haxrrf9YcbMIGwBgsqhkiG9w0BCRACLzGBoDCB
# nTCBmjCBlwQgcl7yf0jhbmm5Y9hCaIxbygeojGkXBkLI/1ord69gXP0wczBfpF0w
# WzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEdsb2JhbFNpZ24gbnYtc2ExMTAvBgNV
# BAMTKEdsb2JhbFNpZ24gVGltZXN0YW1waW5nIENBIC0gU0hBMzg0IC0gRzQCEAEA
# CyAFs5QHYts+NnmUm6kwDQYJKoZIhvcNAQELBQAEggGAQgEUYFgL8eylIbL2GclU
# mUhlsKtQ+yGj661JMLp0tFYRsAcg8JwXCcftuYE5Dmt84z0neUlUkSUmEQ5EZBy7
# PfBC6irag6PS15YL63z04AsRD/BRZPlp3YLH/JA3x2OFPYHu2Uk79WIrBgmcFCGt
# fFTXDpuofwqt0PS9DPwAR+Qk8IgCfxiJbuSDYj75SrWuyZDNqaxcFwEHxAkdeXdI
# zFNHIeGU2+X/VY+5WXavlF0+OCxsdBWdgdAoRYxcvArjQQoUXim/SbRfCQkk9QDH
# 8Mny2zZP8woLWzH+ltNiofV8ioR8Pvu3kt2lz9mbBwuBpHeu8iLRLpB3hj3XNr3V
# O5uJfgWO9b0wCQEwHp3zCIqDC7nFOYARpm08z+zKBChv+dLUI7OR2IiMgwItk8Tt
# 3ds0SnZXeadkE59eQT90OZOzdMmlCpbh5Sxt9tO37hLn4rEOGuUPov9lDtjLS5XF
# 9Tqr40lYGY+SjEO0W6fpMvoODwnM9vZd2TMeXeQCL4Mf
# SIG # End signature block
