<#
.SYNOPSIS
    Locates built in PowerShell Universal variables
.DESCRIPTION
    Checks to make sure that built in PowerShell Universal variables are not being used in the script.
.EXAMPLE
    Measure-PSUBuiltInVariables -ScriptBlockAst $ScriptBlockAst
.INPUTS
    [System.Management.Automation.Language.ScriptBlockAst]
.OUTPUTS
    [Microsoft.Windows.Powershell.ScriptAnalyzer.Generic.DiagnosticRecord[]]
.NOTES
    None
#>
function Measure-PSUBuiltInVariables {
    [CmdletBinding()]
    [OutputType([Microsoft.Windows.Powershell.ScriptAnalyzer.Generic.DiagnosticRecord[]])]
    Param
    (
        [Parameter(Mandatory = $true)]
        [ValidateNotNullOrEmpty()]
        [System.Management.Automation.Language.ScriptBlockAst]
        $ScriptBlockAst
    )

    Process {
        $results = @()

        try {
            #region Define predicates to find ASTs.

            # Finds specific method, IsInRole.
            [ScriptBlock]$predicate1 = {
                param ([System.Management.Automation.Language.Ast]$Ast)

                if ($Ast -is [System.Management.Automation.Language.AssignmentStatementAst]) {
                    if ($Ast.Left -is [System.Management.Automation.Language.VariableExpressionAst]) {   
                        return [PowerShellUniversal.BuiltInVariables2]::IsBuiltInVariable($Ast.Left.VariablePath.UserPath)
                    }
                }

                return $false
            }

            #endregion

            #region Finds ASTs that match the predicates.

            [System.Management.Automation.Language.Ast[]]$methodAst = $ScriptBlockAst.FindAll($predicate1, $true)

            $methodAst | ForEach-Object {
                $result = New-Object `
                    -Typename "Microsoft.Windows.PowerShell.ScriptAnalyzer.Generic.DiagnosticRecord" `
                    -ArgumentList "Overriding built in PowerShell Universal variables can cause undefined behavior.", $_.Extent, $PSCmdlet.MyInvocation.InvocationName, Warning, $null
                $results += $result
            }

            return $results

            #endregion
        }
        catch {
            $PSCmdlet.ThrowTerminatingError($PSItem)
        }
    }
}

<#
.SYNOPSIS
    Locates ConvertTo-Json cmdlets with a depth greater than 5.
.DESCRIPTION
    Checks to make sure that ConvertTo-Json cmdlets are not being used with a depth greater than 5.
.EXAMPLE
    Measure-PSUConvertToJsonDepth -ScriptBlockAst $ScriptBlockAst
.PARAMETER ScriptBlockAst
    The ScriptBlockAst to analyze.
.INPUTS
    [System.Management.Automation.Language.ScriptBlockAst]
.OUTPUTS
    [Microsoft.Windows.Powershell.ScriptAnalyzer.Generic.DiagnosticRecord[]]
.NOTES
    None
#>
function Measure-PSUConvertToJsonDepth {
    [CmdletBinding()]
    [OutputType([Microsoft.Windows.Powershell.ScriptAnalyzer.Generic.DiagnosticRecord[]])]
    Param
    (
        [Parameter(Mandatory = $true)]
        [ValidateNotNullOrEmpty()]
        [System.Management.Automation.Language.ScriptBlockAst]
        $ScriptBlockAst
    )

    Process {
        $results = @()

        try {
            #region Define predicates to find ASTs.

            # Finds specific method, ConvertTo-Json.
            [ScriptBlock]$predicate1 = {
                param ([System.Management.Automation.Language.Ast]$Ast)

                if ($Ast -is [System.Management.Automation.Language.CommandExpressionAst]) {
                    if ($Ast.Expression -is [System.Management.Automation.Language.StringConstantExpressionAst]) {
                        if ($Ast.Expression.Value -eq "ConvertTo-Json") {
                            return $true
                        }
                    }
                }

                return $false
            }

            #endregion
            
            # Finds the depth parameter and checks to see if it is over 5.
            
            [ScriptBlock]$predicate2 = {
                param ([System.Management.Automation.Language.Ast]$Ast)

                if ($Ast -is [System.Management.Automation.Language.CommandParameterAst]) {
                    if ($Ast.ParameterName -eq "Depth") {
                        if ($Ast.Argument -is [System.Management.Automation.Language.ConstantExpressionAst]) {
                            if ($Ast.Argument.Value -gt 5) {
                                return $true
                            }
                        }
                    }
                }

                return $false
            }

            #region Finds ASTs that match the predicates.

            [System.Management.Automation.Language.CommandExpressionAst[]]$methodAst = $ScriptBlockAst.FindAll($predicate1, $true)

            $methodAst | ForEach-Object {
                $_.FindAll($predicate2, $true) | ForEach-Object {
                    $result = New-Object `
                        -Typename "Microsoft.Windows.PowerShell.ScriptAnalyzer.Generic.DiagnosticRecord" `
                        -ArgumentList "ConvertTo-Json should not be used with a depth greater than 5 to avoid performance issues.", $_.Extent, $PSCmdlet.MyInvocation.InvocationName, Warning, $null
                    $results += $result
                }
            }

            return $results

            #endregion
        }
        catch {
            $PSCmdlet.ThrowTerminatingError($PSItem)5
        }
    }
}

Export-ModuleMember -Function Measure-PSUBuiltInVariables
Export-ModuleMember -Function Measure-PSUConvertToJsonDepth