$Cache:SearchIndex = @{}

function New-UDSearchRecord {
    param(
        [string]$Name,
        [string]$Description,
        [string]$Path,
        [string]$Icon
    )
    
    if (-not $Cache:SearchIndex)
    {
        $Cache:SearchIndex = @{}
    }

    $Cache:SearchIndex[$Name] = @{
        Name        = $Name
        Description = $Description
        Path        = $Path
        Icon        = $Icon
    }
}

function Find-UDSearchRecord {
    param(
        [string]$SearchString
    )

    $Cache:SearchIndex.GetEnumerator() | Where-Object {
        $_.Value.Name -like "*$SearchString*" -or $_.Value.Description -like "*$SearchString*"
    } | ForEach-Object {
        $_.Value
    }
}

function New-UDComponentDemo {
    param(
        $Title,
        $Description,
        [string]$Content,
        [string]$Page
    ) 

    if ($Title) {
        $SubTitle = $Description
    }
    elseif ($Description -like "*|*") {
        $Title = $Description.Split("|")[0].Trim()
        $SubTitle = $Description.Split("|")[1].Trim()
    } else {
        $Title = "Example"
        $SubTitle = $Description
    }

    New-UDSearchRecord -Name $Title -Description $SubTitle -Path "/$($Page.Replace(" ", "-"))#$($Title)" -Icon (New-UDIcon -Icon 'Code')
    
    New-UDTypography -Variant "h5" -Text $Title -Id $Title
    New-UDTypography -Variant "subtitle1" -Text $SubTitle

    New-UDPaper -Content {
        New-UDElement -Tag 'div' -Content {} -Attributes @{
            style = @{
                height = "20px"
            }
        }
        & ([ScriptBlock]::Create($Content))
    } -Style @{
        display = "block"
    }

    New-UDPaper -Content {
        New-UDElement -Tag 'div' -Content {
            New-UDStack -Direction column -Content {
                New-UDIconButton -Icon (New-UDIcon -Icon 'Copy') -OnClick {
                    Set-UDClipboard -Data $Content
                    Show-UDToast -Message 'Copied!'
                } -Size small -Style @{
                    float = 'right'
                    width = "30px"
                }
                New-UDSyntaxHighlighter -Code $Content -Style 'oneDark' -ShowLineNumbers
            } 
        } -Attributes @{
            style = @{
                width = "100%"
            }
        }
    }

    New-UDElement -Tag 'div' -Content {} -Attributes @{
        style = @{
            height = "20px"
        }
    }
}

function New-UDComponentPage {
    param($Command, $Title, $Icon, $AdditionalCommands)

    New-UDSearchRecord -Name $Title -Description (Get-Help -Name $Command).SYNOPSIS -Path "/$($Title.Replace(" ", "-"))" -Icon $Icon

    New-UDPage -Name $Title -Icon (New-UDIcon -Icon $Icon) -Id $Command -Content {

        $Help = Get-ChildItem "Cache:$Command" -ErrorAction SilentlyContinue
        if ($Help -eq $null) {
            Set-Item -Path "Cache:$Command" -Value (Get-Help -Name $Command -Full)
            $Help = Get-ChildItem "Cache:$Command" -ErrorAction SilentlyContinue
        }
        
        New-UDTypography -Variant h3 -Text $Title
        New-UDTypography -Variant subtitle1 -Text $Help.SYNOPSIS

        New-UDElement -Tag 'div' -Content {} -Attributes @{
            style = @{
                height = "20px"
            }
        }

        New-UDDivider -Children {
            New-UDTypography -Variant h4 -Text 'Examples'
        }
    
        foreach ($example in $Help.Examples.example) {
            try {
                New-UDComponentDemo -Description $example.remarks[0].text -Content ($example.code.Replace("PS >", "")) -Page $Title
            }
            catch {
                New-UDAlert -Title "Error" -Text "There was an error displaying the example for $Command. $_" -Severity 'Error'
            }
            
        }

        New-UDDivider -Children {
            New-UDTypography -Variant h4 -Text 'Parameters'
        }
    
        $Parameters = $Help.PARAMETERS.Parameter | ForEach-Object {
            [PSCustomObject]@{
                Name         = $_.Name
                Description  = $_.Description.Text
                Type         = $_.Type.name
                Position     = $_.Position
                Required     = $_.Required
                DefaultValue = [string]$_.DefaultValue
            }
        }
    
        New-UDTable -Title $Command -Data $Parameters -Columns @(
            New-UDTableColumn -Property Name -Title 'Name'
            New-UDTableColumn -Property Description -Title 'Description'
            New-UDTableColumn -Property Type -Title 'Type'
            New-UDTableColumn -Property DefaultValue -Title 'Default Value'
            New-UDTableColumn -Property Position -Title 'Position'
            New-UDTableColumn -Property Required -Title 'Required'
        )

        foreach ($additionalCommand in $AdditionalCommands) {
            $CommandHelp = Get-ChildItem "Cache:$additionalCommand" -ErrorAction SilentlyContinue
            if ($CommandHelp -eq $null) {
                Set-Item -Path "Cache:$additionalCommand" -Value (Get-Help -Name $additionalCommand -Full)
                $CommandHelp = Get-ChildItem "Cache:$additionalCommand" -ErrorAction SilentlyContinue
            }

            $Parameters = $CommandHelp.PARAMETERS.Parameter | ForEach-Object {
                [PSCustomObject]@{
                    Name         = $_.Name
                    Description  = $_.Description.Text
                    Type         = $_.Type.name
                    Position     = $_.Position
                    Required     = $_.Required
                    DefaultValue = [string]$_.DefaultValue
                }
            }
        
            New-UDTable -Title $additionalCommand -Data $Parameters -Columns @(
                New-UDTableColumn -Property Name -Title 'Name'
                New-UDTableColumn -Property Description -Title 'Description'
                New-UDTableColumn -Property Type -Title 'Type'
                New-UDTableColumn -Property DefaultValue -Title 'Default Value'
                New-UDTableColumn -Property Position -Title 'Position'
                New-UDTableColumn -Property Required -Title 'Required'
            )
        }
    } 
}

$Pages = @(
    & "$PSScriptRoot\about.ps1"
    & "$PSScriptRoot\pages.ps1"
    & "$PSScriptRoot\host.ps1"
    & "$PSScriptRoot\scopes.ps1"
    & "$PSScriptRoot\theme.ps1"
    & "$PSScriptRoot\variables.ps1"
)

$Pages += New-UDComponentPage -Command 'Add-UDElement' -Title 'Add Element' -Icon 'Plus'
$Pages += New-UDComponentPage -Command 'Clear-UDElement' -Title 'Clear Element' -Icon 'Eraser'
$Pages += New-UDComponentPage -Command 'Invoke-UDJavaScript' -Title 'JavaScript' -Icon 'Js'
$Pages += New-UDComponentPage -Command 'Invoke-UDRedirect' -Title 'Redirect' -Icon 'ArrowUpRightFromSquare'
$Pages += New-UDComponentPage -Command 'New-UDAlert' -Title 'Alert' -Icon 'CircleInfo'
$Pages += New-UDComponentPage -Command 'New-UDAppBar' -Title 'AppBar' -Icon 'Bars'
$Pages += New-UDComponentPage -Command 'New-UDAutocomplete' -Title 'Autocomplete' -Icon 'CaretDown' -AdditionalCommands @("New-UDAutoCompleteOption")
$Pages += New-UDComponentPage -Command 'New-UDAvatar' -Title 'Avatar' -Icon 'User'
$Pages += New-UDComponentPage -Command 'New-UDBackdrop' -Title 'Backdrop' -Icon 'Backward'
$Pages += New-UDComponentPage -Command 'New-UDButton' -Title 'Button' -Icon 'Stop'
$Pages += New-UDComponentPage -Command 'New-UDButtonGroup' -Title 'Button Group' -Icon 'SquareCaretDown' -AdditionalCommands @('New-UDButtonGroupItem')
$Pages += New-UDComponentPage -Command 'New-UDBadge' -Title 'Badge' -Icon 'Certificate'
$Pages += New-UDComponentPage -Command 'New-UDBreadcrumbs' -Title 'Breadcrumbs' -Icon 'BreadSlice'
$Pages += New-UDComponentPage -Command 'New-UDCard' -Title 'Card' -Icon 'AddressCard' -AdditionalCommands @("New-UDCardBody", "New-UDCardHeader", "New-UDCardFooter", "New-UDCardExpand", "New-UDCardMedia")
$Pages += New-UDComponentPage -Command 'New-UDChip' -Title 'Chip' -Icon 'Cookie'
$Pages += New-UDComponentPage -Command 'New-UDChartJS' -Title 'ChartJS' -Icon 'ChartSimple' -AdditionalCommands ("New-UDChartJSDataset", "New-UDChartJSMonitor", "Out-UDChartJSMonitorData")
$Pages += New-UDComponentPage -Command 'New-UDDivider' -Title 'Divider' -Icon 'GridLines'
$Pages += New-UDComponentPage -Command 'New-UDNivoChart' -Title 'Nivo Chart' -Icon 'ChartSimple' -AdditionalCommands ("New-UDNivoChartAxisOptions", "New-UDNivoFill", "New-UDNivoPattern")
$Pages += New-UDComponentPage -Command 'New-UDCheckbox' -Title 'Checkbox' -Icon 'SquareCheck'
$Pages += New-UDComponentPage -Command 'New-UDCodeEditor' -Title 'Code Editor' -Icon 'Code'
$Pages += New-UDComponentPage -Command 'New-UDContainer' -Title 'Container' -Icon 'Box'
$Pages += New-UDComponentPage -Command 'New-UDDataGrid' -Title 'Data Grid' -Icon 'FileSpreadsheet' -AdditionalCommands @("New-UDDataGridColumn", "Out-UDDataGridData")
$Pages += New-UDComponentPage -Command 'New-UDDatePicker' -Title 'Date Picker' -Icon 'Calendar'
$Pages += New-UDComponentPage -Command 'New-UDDateRangePicker' -Title 'Date Range Picker' -Icon 'CalendarDays'
$Pages += New-UDComponentPage -Command 'New-UDDateTime' -Title 'Date and Time' -Icon 'Clock'
$Pages += New-UDComponentPage -Command 'New-UDDivider' -Title 'Divider' -Icon 'Divide'
$Pages += New-UDComponentPage -Command 'New-UDDynamic' -Title 'Dynamic Region' -Icon 'BoltLightning'
$Pages += New-UDComponentPage -Command 'New-UDGauge' -Title 'Gauge' -Icon 'Gauge' 
$Pages += New-UDComponentPage -Command 'New-UDGrid' -Title 'Grid' -Icon 'TableCells'
$Pages += New-UDComponentPage -Command 'New-UDGridLayout' -Title 'Grid Layout' -Icon 'ArrowsUpDownLeftRight'
$Pages += New-UDComponentPage -Command 'New-UDEditor' -Title 'Editor' -Icon 'Paragraph'
$Pages += New-UDComponentPage -Command 'New-UDElement' -Title 'Element' -Icon 'PuzzlePiece'
$Pages += New-UDComponentPage -Command 'New-UDErrorBoundary' -Title 'Error Boundary' -Icon 'CircleExclamation'
$Pages += New-UDComponentPage -Command 'New-UDExpansionPanelGroup' -Title 'Expansion Panel' -Icon 'Expand' -AdditionalCommands "New-UDExpansionPanel"
$Pages += New-UDComponentPage -Command 'New-UDFloatingActionButton' -Title 'Floating Action Button' -Icon 'Ghost'
$Pages += New-UDComponentPage -Command 'New-UDForm' -Title 'Form' -Icon 'RectangleList' -AdditionalCommands "New-UDValidationResult"
$Pages += New-UDComponentPage -Command 'New-UDHidden' -Title 'Hidden' -Icon 'EyeSlash'
$Pages += New-UDComponentPage -Command 'New-UDHtml' -Title 'HTML' -Icon 'Code'
$Pages += New-UDComponentPage -Command 'New-UDIcon' -Title 'Icon' -Icon 'FontAwesome' -AdditionalCommands "Find-UDIcon"
$Pages += New-UDComponentPage -Command 'New-UDIconButton' -Title 'Icon Button' -Icon 'CircleDot' 
$Pages += New-UDComponentPage -Command 'New-UDImage' -Title 'Image' -Icon 'Image'
$Pages += New-UDComponentPage -Command 'New-UDLayout' -Title 'Layout' -Icon 'BorderNone'
$Pages += New-UDComponentPage -Command 'New-UDLink' -Title 'Link' -Icon 'Link'
$Pages += New-UDComponentPage -Command 'New-UDList' -Title 'List' -Icon 'List' -AdditionalCommands 'New-UDListItem'
$Pages += New-UDComponentPage -Command 'New-UDMap' -Title 'Map' -Icon 'Map' -AdditionalCommands @('New-UDMapBaseLayer', 'New-UDMapRasterLayer', "New-UDMapVectorLayer", 'New-UDMapMarker', "New-UDMapFeatureGroup", "New-UDMapLayerControl", 'New-UDMapPopup', 'New-UDMapIcon')
$Pages += New-UDComponentPage -Command 'New-UDMarkdown' -Title 'Markdown' -Icon 'Markdown'
$Pages += New-UDComponentPage -Command 'New-UDMenu' -Title 'Menu' -Icon 'EllipsisV' -AdditionalCommands 'New-UDMenuItem'
$Pages += New-UDComponentPage -Command 'New-UDPaper' -Title 'Paper' -Icon 'Newspaper'
$Pages += New-UDComponentPage -Command 'New-UDProgress' -Title 'Progress' -Icon 'Spinner'
$Pages += New-UDComponentPage -Command "Protect-UDSection" -Title 'Protect Section' -Icon 'Lock'
$Pages += New-UDComponentPage -Command 'New-UDRadioGroup' -Title 'Radio' -Icon 'Lock' -AdditionalCommands 'New-UDRadio'
$Pages += New-UDComponentPage -Command 'New-UDRating' -Title 'Rating' -Icon 'Star'
$Pages += New-UDComponentPage -Command 'New-UDSelect' -Title 'Select' -Icon 'RectangleList' -AdditionalCommands @( 'New-UDSelectOption', 'New-UDSelectGroup')
$Pages += New-UDComponentPage -Command 'New-UDSkeleton' -Title 'Skeleton' -Icon 'BarsProgress'
$Pages += New-UDComponentPage -Command 'New-UDSlider' -Title 'Slider' -Icon 'Sliders'
$Pages += New-UDComponentPage -Command 'New-UDSparkline' -Title 'Sparkline' -Icon 'ChartLine'
$Pages += New-UDComponentPage -Command 'New-UDSpeedDial' -Title 'Speed Dial' -Icon 'TruckFast' -AdditionalCommands @("New-UDSpeedDialAction")
$Pages += New-UDComponentPage -Command 'New-UDStack' -Title 'Stack' -Icon 'Cubes'
$Pages += New-UDComponentPage -Command 'New-UDStepper' -Title 'Stepper' -Icon 'ForwardStep' -AdditionalCommands @("New-UDStep")
$Pages += New-UDComponentPage -Command 'New-UDStyle' -Title 'Style' -Icon 'PaintRoller'
$Pages += New-UDComponentPage -Command 'New-UDSwitch' -Title 'Switch' -Icon 'ToggleOn'
$Pages += New-UDComponentPage -Command 'New-UDSyntaxHighlighter' -Title 'Syntax Highlighter' -Icon 'Code'
$Pages += New-UDComponentPage -Command 'New-UDTabs' -Title 'Tab' -Icon 'Tablet' -AdditionalCommands "New-UDTab"
$Pages += New-UDComponentPage -Command 'New-UDTable' -Title 'Table' -Icon 'Table' -AdditionalCommands @("New-UDTableColumn", "New-UDTableTextOption", "Out-UDTableData")
$Pages += New-UDComponentPage -Command 'New-UDTextbox' -Title 'Textbox' -Icon 'Keyboard'
$Pages += New-UDComponentPage -Command 'New-UDTimeline' -Title 'Timeline' -Icon 'Timeline' -AdditionalCommands @("New-UDTimelineItem")
$Pages += New-UDComponentPage -Command 'New-UDTimePicker' -Title 'Time Picker' -Icon 'Clock'
$Pages += New-UDComponentPage -Command 'New-UDToggleButtonGroup' -Title 'Toggle Button' -Icon 'MattressPillow' -AdditionalCommands @("New-UDToggleButton")
$Pages += New-UDComponentPage -Command 'New-UDTooltip' -Title 'Tooltip' -Icon 'Readme'
$Pages += New-UDComponentPage -Command 'New-UDTransferList' -Title 'Transfer List' -Icon 'RightLeft' -AdditionalCommands @("New-UDTransferListItem")
$Pages += New-UDComponentPage -Command 'New-UDTransition' -Title 'Transition' -Icon 'HatWizard'
$Pages += New-UDComponentPage -Command 'New-UDTreeView' -Title 'Tree View' -Icon 'Tree' -AdditionalCommands @("New-UDTreeNode")
$Pages += New-UDComponentPage -Command 'New-UDTypography' -Title 'Typography' -Icon 'Font'
$Pages += New-UDComponentPage -Command 'New-UDUpload' -Title 'Upload' -Icon 'Upload'
$Pages += New-UDComponentPage -Command 'Remove-UDElement' -Title 'Remove Element' -Icon 'Trash' 
$Pages += New-UDComponentPage -Command 'Set-UDClipboard' -Title 'Clipboard' -Icon 'Clipboard' 
$Pages += New-UDComponentPage -Command 'Set-UDElement' -Title 'Set Element' -Icon 'Pen' 
$Pages += New-UDComponentPage -Command 'Show-UDModal' -Title 'Modal' -Icon 'WindowMaximize' -AdditionalCommands 'Hide-UDModal'
$Pages += New-UDComponentPage -Command 'Show-UDSnackbar' -Title 'Snackbar' -Icon 'CookieBite' 
$Pages += New-UDComponentPage -Command 'Show-UDToast' -Title 'Toast' -Icon 'Comment' -AdditionalCommands 'Hide-UDToast'
$Pages += New-UDComponentPage -Command 'Start-UDDownload' -Title 'Download' -Icon 'Download'
$Pages += New-UDComponentPage -Command 'Sync-UDElement' -Title 'Sync Element' -Icon 'Rotate'

$Navigation = @(
    New-UDListItem -Label 'About' -Href '/About' -Icon (New-UDIcon -Icon 'CircleQuestion')
    New-UDListItem -Label 'Inputs' -Children {
        New-UDListItem -Label 'Autocomplete' -Href '/Autocomplete' -Icon (New-UDIcon -Icon 'CaretDown')  -Nested
        New-UDListItem -Label 'Button' -Href '/Button' -Icon (New-UDIcon -Icon 'Stop') -Nested
        New-UDListItem -Label 'Button Group' -Href '/Button-Group' -Icon (New-UDIcon -Icon 'SquareCaretDown') -Nested
        New-UDListItem -Label 'Checkbox' -Href '/Checkbox' -Icon (New-UDIcon -Icon 'SquareCheck') -Nested
        New-UDListItem -Label 'Code Editor' -Href '/Code-Editor' -Icon (New-UDIcon -Icon 'Code') -Nested
        New-UDListItem -Label 'Date Picker' -Href '/Date-Picker' -Icon (New-UDIcon -Icon 'Calendar') -Nested
        New-UDListItem -Label 'Date Range Picker' -Href '/Date-Range-Picker' -Icon (New-UDIcon -Icon 'CalendarDays') -Nested
        New-UDListItem -Label 'Editor' -Href '/Editor' -Icon (New-UDIcon -Icon 'Paragraph') -Nested
        New-UDListItem -Label 'Floating Action Button' -Href '/Floating-Action-Button' -Icon (New-UDIcon -Icon 'Ghost') -Nested
        New-UDListItem -Label 'Form' -Href '/Form' -Icon (New-UDIcon -Icon 'RectangleList') -Nested
        New-UDListItem -Label 'Icon Button' -Href '/Icon-Button' -Icon (New-UDIcon -Icon 'CircleDot') -Nested
        New-UDListItem -Label 'Radio' -Href '/Radio' -Icon (New-UDIcon -Icon 'CircleDot') -Nested
        New-UDListItem -Label 'Rating' -Href '/Rating' -Icon (New-UDIcon -Icon 'Star') -Nested
        New-UDListItem -Label 'Select' -Href '/Select' -Icon (New-UDIcon -Icon 'RectangleList') -Nested
        New-UDListItem -Label 'Slider' -Href '/Slider' -Icon (New-UDIcon -Icon 'Sliders') -Nested
        New-UDListItem -Label 'Speed Dial' -Href '/Speed-Dial' -Icon (New-UDIcon -Icon 'TruckFast') -Nested
        New-UDListItem -Label 'Switch' -Href '/Switch' -Icon (New-UDIcon -Icon 'ToggleOn') -Nested
        New-UDListItem -Label 'Textbox' -Href '/Textbox' -Icon (New-UDIcon -Icon 'Keyboard') -Nested
        New-UDListItem -Label 'Toggle Button' -Href '/Toggle-Button' -Icon (New-UDIcon -Icon 'MattressPillow') -Nested
        New-UDListItem -Label 'Time Picker' -Href '/Time-Picker' -Icon (New-UDIcon -Icon 'Clock') -Nested
        New-UDListItem -Label 'Transfer List' -Href '/Transfer-List' -Icon (New-UDIcon -Icon 'RightLeft') -Nested
        New-UDListItem -Label 'Upload' -Href '/Upload' -Icon (New-UDIcon -Icon 'Upload') -Nested
    }
    New-UDListItem -Label 'Data Display' -Children {
        New-UDListItem -Label 'Avatar' -Href '/Avatar' -Icon (New-UDIcon -Icon 'User') -Nested
        New-UDListItem -Label 'Badge' -Href '/Badge' -Icon (New-UDIcon -Icon 'Certificate') -Nested
        New-UDListItem -Label 'ChartJS' -Href '/ChartJS' -Icon (New-UDIcon -Icon 'ChartSimple') -Nested
        New-UDListItem -Label 'Chip' -Href '/Chip' -Icon (New-UDIcon -Icon 'Cookie') -Nested
        New-UDListItem -Label 'Data Grid' -Href '/Data-Grid' -Icon (New-UDIcon -Icon 'table-cells') -Nested
        New-UDListItem -Label 'Data and Time' -Href '/Date-and-Time' -Icon (New-UDIcon -Icon 'Clock') -Nested
        New-UDListItem -Label 'Gauge' -Href '/Gauge' -Icon (New-UDIcon -Icon 'Gauge') -Nested
        New-UDListItem -Label 'Icon' -Href '/Icon' -Icon (New-UDIcon -Icon 'FontAwesome') -Nested
        New-UDListItem -Label 'Image' -Href '/Image' -Icon (New-UDIcon -Icon 'Image') -Nested
        New-UDListItem -Label 'Link' -Href '/Link' -Icon (New-UDIcon -Icon 'Link') -Nested
        New-UDListItem -Label 'List' -Href '/List' -Icon (New-UDIcon -Icon 'List') -Nested
        New-UDListItem -Label 'Map' -Href '/Map' -Icon (New-UDIcon -Icon 'Map') -Nested
        New-UDListItem -Label 'Markdown' -Href '/Markdown' -Icon (New-UDIcon -Icon 'Markdown') -Nested
        New-UDListItem -Label 'Nivo Chart' -Href '/Nivo-Chart' -Icon (New-UDIcon -Icon 'ChartSimple') -Nested
        New-UDListItem -Label 'Sparkline' -Href '/Sparkline' -Icon (New-UDIcon -Icon 'ChartLine') -Nested
        New-UDListItem -Label 'Syntax Highlighter' -Href '/Syntax-Highlighter' -Icon (New-UDIcon -Icon 'Code') -Nested
        New-UDListItem -Label 'Table' -Href '/Table' -Icon (New-UDIcon -Icon 'Table') -Nested
        New-UDListItem -Label 'Timeline' -Href '/Timeline' -Icon (New-UDIcon -Icon 'Timeline') -Nested
        New-UDListItem -Label 'Typography' -Href '/Typography' -Icon (New-UDIcon -Icon 'Font') -Nested
    }
    New-UDListItem -Label 'Feedback' -Children {
        New-UDListItem -Label 'Alert' -Href '/Alert' -Icon (New-UDIcon -Icon 'CircleExclamation') -Nested
        New-UDListItem -Label 'Backdrop' -Href '/Backdrop' -Icon (New-UDIcon -Icon 'BoothCurtain') -Nested
        New-UDListItem -Label 'Progress' -Href '/Progress' -Icon (New-UDIcon -Icon 'Spinner') -Nested
        New-UDListItem -Label 'Skeleton' -Href '/Skeleton' -Icon (New-UDIcon -Icon 'BarsProgress') -Nested
    }
    New-UDListItem -Label 'Surfaces'-Children {
        New-UDListItem -Label 'AppBar' -Href '/AppBar' -Icon (New-UDIcon -Icon 'Bars') -Nested
        New-UDListItem -Label 'Card' -Href '/Card' -Icon (New-UDIcon -Icon 'AddressCard') -Nested
        New-UDListItem -Label 'Expansion Panel' -Href '/Expansion-Panel' -Icon (New-UDIcon -Icon 'Expand') -Nested
        New-UDListItem -Label 'Paper' -Href '/Paper' -Icon (New-UDIcon -Icon 'Newspaper') -Nested
    }
    New-UDListItem -Label 'Navigation' -Children {
        New-UDListItem -Label 'Breadcrumbs' -Href '/Breadcrumbs' -Icon (New-UDIcon -Icon 'BreadSlice') -Nested
        New-UDListItem -Label 'Menu' -Href '/Menu' -Icon (New-UDIcon -Icon 'EllipsisV') -Nested
        New-UDListItem -Label 'Page' -Href '/Page' -Icon (New-UDIcon -Icon 'File') -Nested
        New-UDListItem -Label 'Stepper' -Href '/Stepper' -Icon (New-UDIcon -Icon 'ForwardStep') -Nested
        New-UDListItem -Label 'Tab' -Href '/Tab' -Icon (New-UDIcon -Icon 'Tablet') -Nested
        New-UDListItem -Label 'Tree View' -Href '/Tree-View' -Icon (New-UDIcon -Icon 'Tree') -Nested
    }
    New-UDListItem -Label 'Layout' -Children {
        New-UDListItem -Label 'Container' -Href '/Container' -Icon (New-UDIcon -Icon 'Box') -Nested
        New-UDListItem -Label 'Divider' -Href '/Divider' -Icon (New-UDIcon -Icon 'HorizontalRule') -Nested
        New-UDListItem -Label 'Hidden' -Href '/Hidden' -Icon (New-UDIcon -Icon 'EyeSlash') -Nested
        New-UDListItem -Label 'Grid' -Href '/Grid' -Icon (New-UDIcon -Icon 'TableCells') -Nested
        New-UDListItem -Label 'Grid Layout' -Href '/Grid-Layout' -Icon (New-UDIcon -Icon 'TablePivot') -Nested
        New-UDListItem -Label 'Layout' -Href '/Layout' -Icon (New-UDIcon -Icon 'TableLayout') -Nested
        New-UDListItem -Label 'Stack' -Href '/Stack' -Icon (New-UDIcon -Icon 'Cubes') -Nested
    }
    New-UDListItem -Label 'Interaction' -Children {
        New-UDListItem -Label 'Add Element' -Href '/Add-Element' -Icon (New-UDIcon -Icon 'Plus') -Nested
        New-UDListItem -Label 'Clear Element' -Href '/Clear-Element' -Icon (New-UDIcon -Icon 'Eraser') -Nested
        New-UDListItem -Label 'Clipboard' -Href '/Clipboard' -Icon (New-UDIcon -Icon 'Clipboard') -Nested
        New-UDListItem -Label 'Download' -Href '/Download' -Icon (New-UDIcon -Icon 'Download') -Nested
        New-UDListItem -Label 'JavaScript' -Href '/JavaScript' -Icon (New-UDIcon -Icon 'Js') -Nested
        New-UDListItem -Label 'PowerShell Host' -Href '/PowerShell-Host' -Icon (New-UDIcon -Icon 'SquareTerminal') -Nested
        New-UDListItem -Label 'Redirect' -Href '/Redirect' -Icon (New-UDIcon -Icon 'ArrowUpRightFromSquare') -Nested
        New-UDListItem -Label 'Remove Element' -Href '/Remove-Element' -Icon (New-UDIcon -Icon 'Trash') -Nested
        New-UDListItem -Label 'Set Element' -Href '/Set-Element' -Icon (New-UDIcon -Icon 'Pen') -Nested
        New-UDListItem -Label 'Sync Element' -Href '/Sync-Element' -Icon (New-UDIcon -Icon 'Rotate') -Nested
    }
    New-UDListItem -Label 'Styling' -Children {
        New-UDListItem -Label 'Theme' -Href '/Theme' -Icon (New-UDIcon -Icon 'Paintbrush') -Nested
        New-UDListItem -Label 'Style' -Href '/Style' -Icon (New-UDIcon -Icon 'PaintRoller') -Nested
    }
    New-UDListItem -Label 'Utilities' -Children {
        New-UDListItem -Label 'Element' -Href '/Element' -Icon (New-UDIcon -Icon 'PuzzlePiece') -Nested
        New-UDListItem -Label 'Error Boundary' -Href '/Error-Boundary' -Icon (New-UDIcon -Icon 'CircleExclamation') -Nested
        New-UDListItem -Label 'Dynamic Region' -Href '/Dynamic-Region' -Icon (New-UDIcon -Icon 'BoltLightning') -Nested
        New-UDListItem -Label 'HTML' -Href '/HTML' -Icon (New-UDIcon -Icon 'Code') -Nested
        New-UDListItem -Label 'Modal' -Href '/Modal' -Icon (New-UDIcon -Icon 'WindowMaximize') -Nested
        New-UDListItem -Label 'Protect Section' -Href '/Protect-Section' -Icon (New-UDIcon -Icon 'Lock') -Nested
        New-UDListItem -Label 'Scopes' -Href '/Scopes' -Icon (New-UDIcon -Icon 'UserGear') -Nested
        New-UDListItem -Label 'Snackbar' -Href '/Snackbar' -Icon (New-UDIcon -Icon 'CookieBite') -Nested
        New-UDListItem -Label 'Toast' -Href '/Toast' -Icon (New-UDIcon -Icon 'Comment') -Nested
        New-UDListItem -Label 'Tooltip' -Href '/Tooltip' -Icon (New-UDIcon -Icon 'Readme') -Nested
        New-UDListItem -Label 'Transition' -Href '/Transition' -Icon (New-UDIcon -Icon 'Transporter3') -Nested
    }
    New-UDListItem -Label 'Variables' -Href '/Variables' -Icon (New-UDIcon -Icon 'SquareRootVariable') 
)

New-UDDashboard -Title 'PowerShell Universal' -NavigationLayout permanent -Navigation $Navigation -Pages $Pages -Menu {
    New-UDMenuItem -Text 'Profile' -Icon (New-UDIcon -Icon 'User') 
} -HeaderContent {
    New-UDAutocomplete -OnLoadOptions {
        Find-UDSearchRecord -SearchString $Body | ForEach-Object {
            New-UDAutoCompleteOption -Name $_.Name -Value $_.Path
        }
    } -OnChange {
        Invoke-UDRedirect -Url $Body
    } -Label "Search" -Icon (New-UDIcon -Icon 'Search') 
}